import express from 'express';
import bodyParser from 'body-parser';
import morgan from 'morgan';
import pino from 'pino';
import { CFG } from './config.js';
import { loadState } from './storage.js';
import { engine } from './engine.js';

const log = pino({ transport: { target: 'pino-pretty' } });
const app = express();
app.use(morgan('tiny'));
app.use(bodyParser.json());

function auth(req, res, next){
  const tok = req.headers['x-auth-token'];
  if (!CFG.TOKEN || tok !== CFG.TOKEN) return res.status(401).json({ error: 'unauthorized' });
  next();
}

function computeFiltersOk(payload){
  const { adx14=0, diplus=0, diminus=0, swingHigh10=0, price=0, ema20=0 } = payload;
  const adxPrevRising = true; // if you can pass prev ADX from TV, tighten this
  const adxOk = adx14 > 20 && adxPrevRising;
  const diOk = diplus > diminus;
  const swingOk = price > (swingHigh10 || 0);
  const emaOk = price > (ema20 || 0);
  return adxOk && diOk && swingOk && emaOk;
}

app.post('/webhook', auth, async (req, res) => {
  const payload = req.body || {};
  const state = loadState();
  try {
    if (payload.type === 'ENTRY') {
      const r = engine.onEntry(state, payload);
      return res.json({ ok: true, result: r });
    }
    if (payload.type === 'TICK') {
      payload.filtersOk = computeFiltersOk(payload);
      const r = engine.onTick(state, payload);
      return res.json({ ok: true, result: r });
    }
    return res.status(400).json({ error: 'unknown type' });
  } catch (e) {
    log.error(e);
    return res.status(500).json({ error: 'internal', detail: String(e) });
  }
});

app.get('/state', (req,res)=>{ res.json(loadState()); });

app.post('/reset', auth, (req,res)=>{
  const s = { symbol: CFG.SYMBOL, equityUSDT: 3800, pos: null, cooldownUntilBar: -1, logs: [] };
  import('./storage.js').then(({ saveState }) => { saveState(s); res.json({ ok:true }); });
});

const port = process.env.PORT || CFG.PORT || 3000;
app.listen(port, ()=>{
  log.info(`Bot server on :${port} PAPER=${CFG.PAPER}`);
});
